-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModPath = g_currentModDirectory
source(thModPath .. "scripts/core/utilities/THUtils.lua")
source(thModPath .. "scripts/core/utilities/THDebugUtil.lua")
source(thModPath .. "scripts/core/managers/THModManager.lua")
source(thModPath .. "scripts/core/managers/THEventManager.lua")
_G.THSpecType = {
    VEHICLE = 1,
    PLACEABLE = 2
}
THCore = {}
local THCore_mt = THUtils.createClass(THCore)
function THCore.registerXMLPaths(xmlSchema, xmlPath)
end
function THCore.new(name, className, modName, customMt)
    customMt = customMt or THCore_mt
    if THUtils.argIsValid(THUtils.validateId(name) == true, "name", name)
        and THUtils.argIsValid(type(className) == THValueType.STRING, "className", className)
        and THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        local classTable = _G[className]
        local modData = g_thModManager:getLoadedMod(modName)
        local modEnv = nil
        if modData ~= nil then
            modEnv = modData.environment
        end
        if type(classTable) ~= THValueType.TABLE then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "className", className)
        elseif modData == nil or type(modEnv) ~= THValueType.TABLE then
            THUtils.errorMsg(true, "Cannot find %q mod environment")
        else
            local self = setmetatable({}, customMt)
            self.modData = modData
            self.modName = modData.name
            self.modPath = modData.path
            self.isServer = g_server ~= nil
            self.isClient = g_client ~= nil
            self.name = name
            self.xmlKey = self.name
            self.dataKey = self.modName .. "." .. self.name
            self.thModManager = g_thModManager
            self.thEventManager = g_thEventManager
            self.hasMapConfig = false
            local mapSettingsModName = string.format("FS%d_TH_MapSettings", THGameVersion)
            local mapSettingsModData = self.thModManager:getLoadedMod(mapSettingsModName)
            if mapSettingsModData ~= nil then
                self.thMapSettings = {
                    modName = mapSettingsModData.name,
                    modPath = mapSettingsModData.path,
                    modData = mapSettingsModData,
                    xmlFilename = THUtils.getFilename("maps/mapSettings.xml", mapSettingsModData.path),
                    xmlDataKey = "thMapSettings",
                    xmlRootKey = "thMapSettings"
                }
            end
            self.i3DManager = modEnv.g_i3DManager
            self.inputManager = modEnv.g_inputBinding
            self.taskManager = modEnv.g_asyncTaskManager
            self.i18n = modEnv.g_i18n
            self.specTypes = {}
            self.specTypeMapping = {}
            self.specializations = {}
            for typeId, typeIndex in pairs(THSpecType) do
                local specTypeData = {
                    id = typeId,
                    index = typeIndex,
                    name = "object",
                    namePlural = "objects"
                }
                if typeIndex == THSpecType.VEHICLE then
                    specTypeData.name = "vehicle"
                    specTypeData.namePlural = "vehicles"
                    self.specTypeMapping[specTypeData.name] = typeIndex
                elseif typeIndex == THSpecType.PLACEABLE then
                    specTypeData.name = "placeable"
                    specTypeData.namePlural = "placeables"
                    self.specTypeMapping[specTypeData.name] = typeIndex
                end
                self.specTypes[typeIndex] = specTypeData
                self.specializations[typeIndex] = {}
            end
            self.thirdPartyMods = {}
            local pfModData = self.thModManager:getLoadedMod(THModManager.MOD.PRECISION_FARMING)
            if pfModData ~= nil and pfModData.environment ~= nil and pfModData.target ~= nil then
                local PrecisionFarming = pfModData.environment.PrecisionFarming
                if PrecisionFarming ~= nil then
                    self.thirdPartyMods.precisionFarming = pfModData
                end
            end
            self.thEventManager:addEventListener(self)
            THDebugUtil.createConsoleCommands(self)
            return self
        end
    end
end
function THCore.onSetMissionInfo(self, mission, missionInfo, missionDynamicInfo)
    self.mission = mission
    self.missionInfo = missionInfo
    if missionInfo ~= nil and missionInfo.map ~= nil then
        local mapDesc = missionInfo.map
        if type(mapDesc.id) == THValueType.STRING and mapDesc.id ~= ""
            and type(mapDesc.mapXMLFilename) == THValueType.STRING and mapDesc.mapXMLFilename ~= ""
        then
            local mapData = {
                id = mapDesc.id:upper(),
                name = mapDesc.id,
                xmlFilename = mapDesc.mapXMLFilename,
                info = mapDesc,
                isModMap = mapDesc.isModMap == true
            }
            if type(mapDesc.baseDirectory) == THValueType.STRING and mapDesc.baseDirectory ~= "" then
                mapData.path = mapDesc.baseDirectory
            end
            if type(mapDesc.customEnvironment) == THValueType.STRING and mapDesc.customEnvironment ~= "" then
                mapData.modName = mapDesc.customEnvironment
            end
            local absXMLFilename = THUtils.getFilename(mapData.xmlFilename, mapData.path)
            if THUtils.getFileExists(absXMLFilename) then
                self.mapData = mapData
            else
                THUtils.errorMsg(nil, THMessage.FILE_NOT_FOUND, absXMLFilename)
            end
        end
    end
end
function THCore.onFinalizeTypes(self, manager)
    local specTypeData = self.specTypes[self.specTypeMapping[manager.typeName]]
    if specTypeData ~= nil then
        local specsArray = self.specializations[specTypeData.index]
        local typedObjects = manager:getTypes()
        if specsArray ~= nil and typedObjects ~= nil then
            for specIndex = 1, #specsArray do
                local specData = specsArray[specIndex]
                local timesAdded = 0
                for objectTypeName, objectType in pairs(typedObjects) do
                    if objectType.specializations ~= nil
                        and not SpecializationUtil.hasSpecialization(specData.class, objectType.specializations)
                        and specData.class.prerequisitesPresent(objectType.specializations)
                    then
                        manager:addSpecialization(objectTypeName, specData.absName)
                        if SpecializationUtil.hasSpecialization(specData.class, objectType.specializations) then
                            timesAdded = timesAdded + 1
                        end
                    end
                end
                if timesAdded > 0 then
                    THUtils.displayMsg("Specialization %q added to %d %s", specData.name, timesAdded, specTypeData.namePlural)
                end
            end
        end
    end
end
function THCore.getMapXMLFilename(self, xmlKey, useMapSettings, forceMapSettings)
    if THUtils.argIsValid(xmlKey == nil or (type(xmlKey) == THValueType.STRING and xmlKey ~= ""), "xmlKey", xmlKey)
        and THUtils.argIsValid(not useMapSettings or useMapSettings == true, "useMapSettings", useMapSettings)
        and THUtils.argIsValid(not forceMapSettings or forceMapSettings == true, "forceMapSettings", forceMapSettings)
        and self.mapData ~= nil
    then
        local mapData = self.mapData
        local mapXMLFilename = nil
        local customEnv = nil
        local baseDirectory = nil
        local isMapSettings = false
        if useMapSettings then
            local thMapSettings = self.thMapSettings
            if thMapSettings ~= nil and fileExists(thMapSettings.xmlFilename) then
                local settingsXMLFile = THUtils.loadXMLFile("THMapSettingsXML", thMapSettings.xmlFilename, nil, true)
                local settingsXMLKey = thMapSettings.xmlRootKey .. ".maps." .. mapData.name
                if settingsXMLFile ~= nil then
                    if THUtils.hasXMLProperty(settingsXMLFile, settingsXMLKey) then
                        mapXMLFilename = THUtils.getXMLValue(settingsXMLFile, XMLValueType.STRING, settingsXMLKey, "#filename")
                        if mapXMLFilename ~= nil then
                            mapXMLFilename = THUtils.getFilename(mapXMLFilename, thMapSettings.modPath)
                            if THUtils.getFileExists(mapXMLFilename) then
                                customEnv = thMapSettings.modName
                                baseDirectory = thMapSettings.modPath
                                isMapSettings = true
                            else
                                mapXMLFilename = nil
                            end
                        end
                    end
                    THUtils.deleteXMLFile(settingsXMLFile)
                end
            end
        end
        if mapXMLFilename == nil and not forceMapSettings and mapData.isModMap then
            mapXMLFilename = THUtils.getFilename(mapData.xmlFilename, mapData.path, true)
            if mapXMLFilename ~= nil then
                if THUtils.getFileExists(mapXMLFilename) then
                    customEnv = mapData.modName
                    baseDirectory = mapData.path
                else
                    mapXMLFilename = nil
                end
            end
        end
        if mapXMLFilename ~= nil then
            local dataXMLFilename = nil
            local dataXMLKey = "map"
            if xmlKey == nil then
                dataXMLFilename = mapXMLFilename
            else
                dataXMLKey = "map." .. xmlKey
                local mapXMLFile = THUtils.loadXMLFile("THMapXML", mapXMLFilename, nil, true)
                if mapXMLFile ~= nil
                    and THUtils.hasXMLProperty(mapXMLFile, dataXMLKey)
                then
                    dataXMLFilename = THUtils.getXMLValue(mapXMLFile, XMLValueType.STRING, dataXMLKey, "#filename")
                    if dataXMLFilename == nil then
                        dataXMLFilename = mapXMLFilename
                    else
                        dataXMLFilename = THUtils.getFilename(dataXMLFilename, baseDirectory)
                        if THUtils.getFileExists(dataXMLFilename) then
                            dataXMLKey = xmlKey
                        else
                            dataXMLFilename = nil
                        end
                    end
                    THUtils.deleteXMLFile(mapXMLFile)
                end
            end
            if dataXMLFilename ~= nil then
                return dataXMLFilename, dataXMLKey, customEnv, baseDirectory, isMapSettings
            end
        end
    end
end
function THCore.loadMapXMLFile(self, xmlName, xmlKey, xmlSchema, useMapSettings, forceMapSettings)
    useMapSettings = THUtils.getNoNil(useMapSettings, true)
    if THUtils.argIsValid(type(xmlName) == THValueType.STRING and xmlName ~= "", "xmlName", xmlName)
        and THUtils.argIsValid(not xmlSchema or xmlSchema == true or type(xmlSchema) == THValueType.TABLE, "xmlSchema", xmlSchema)
    then
        local dataXMLFilename, dataXMLKey, customEnv, baseDirectory, isMapSettings = self:getMapXMLFilename(xmlKey, useMapSettings, forceMapSettings)
        if dataXMLFilename ~= nil then
            local dataXMLFile = THUtils.loadXMLFile(xmlName, dataXMLFilename, nil, xmlSchema)
            if dataXMLFile ~= nil then
                return dataXMLFile, dataXMLFilename, dataXMLKey, customEnv, baseDirectory, isMapSettings
            end
        end
    end
end
function THCore.loadMapData(self)
    self.hasMapConfig = false
    local mapXMLFile, mapXMLFilename, mapXMLKey, customEnv, baseDirectory = self:loadMapXMLFile("THMapConfigXML", self.xmlKey, self.xmlSchema, true)
    local success = true
    if mapXMLFile ~= nil and mapXMLKey ~= nil then
        THUtils.displayMsg(THMessage.LOADING, mapXMLFilename)
        success = self:loadFromMapXML(mapXMLFile, mapXMLKey, customEnv, baseDirectory)
        if success then
            self.hasMapConfig = true
        else
            THUtils.errorMsg(nil, THMessage.LOADING_ERROR, mapXMLFilename)
        end
        THUtils.deleteXMLFile(mapXMLFile)
    end
    return success
end
function THCore.loadFromMapXML(self, xmlFile, xmlKey, customEnv, baseDirectory)
    customEnv = customEnv or self.modName
    baseDirectory = baseDirectory or self.modPath
    if THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(type(customEnv) == THValueType.STRING, "customEnv", customEnv)
        and THUtils.argIsValid(type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        return true
    end
    return false
end
function THCore.addSpecialization(self, specType, specName, className, modName, filename)
    modName = modName or self.modName
    local modData = self.thModManager:getLoadedMod(modName)
    local specTypeData = nil
    if type(specType) == THValueType.STRING then
        local specTypeIndex = self.specTypeMapping[specType]
        if specTypeIndex ~= nil then
            specTypeData = self.specTypes[specTypeIndex]
        end
    elseif specType ~= nil then
        specTypeData = self.specTypes[specType]
    end
    if THUtils.argIsValid(specTypeData ~= nil, "specType", specType)
        and THUtils.argIsValid(type(specName) == THValueType.STRING and specName ~= "", "specName", specName)
        and THUtils.argIsValid(type(className) == THValueType.STRING, "className", className)
        and THUtils.argIsValid(modData ~= nil, "modName", modName)
        and THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
    then
        local manager = nil
        if specTypeData.index == THSpecType.VEHICLE then
            manager = g_specializationManager
        elseif specTypeData.index == THSpecType.PLACEABLE then
            manager = g_placeableSpecializationManager
        else
            THUtils.errorMsg(nil, "Could not find %s specialization manager", specTypeData.name)
            return false
        end
        if manager ~= nil then
            local absSpecName = modName .. "." .. specName
            local absClassName = modName .. "." .. className
            local absFilename = Utils.getFilename(filename, self.modPath)
            if absFilename == nil or not fileExists(absFilename) then
                THUtils.errorMsg(nil, THMessage.FILE_NOT_FOUND, absFilename)
                return false
            end
            if manager:getSpecializationByName(specName) then
                THUtils.errorMsg(nil, "Duplicate %s specialization found: %s", specTypeData.name, specName)
                return false
            end
            local specData = {
                name = specName,
                absName = absSpecName,
                className = className,
                absClassName = absClassName,
                filename = filename,
                absFilename = absFilename,
                specType = specTypeData,
                modName = modName
            }
            manager:addSpecialization(specName, className, absFilename, modName)
            if manager:getSpecializationByName(specName) then
                local specClass = _G[className]
                if specClass == nil then
                    THUtils.errorMsg(nil, "Cannot find specialization %q class table: %s", specName, className)
                    return false
                end
                specData.class = specClass
                table.insert(self.specializations[specTypeData.index], specData)
                THUtils.displayMsg("Added %s specialization: %s", specTypeData.name, specName)
                return true
            end
        end
    end
    return false
end
function THCore.consoleCommandSetDebugEnabled(self, flagName, isEnabled, sendToServer)
    THUtils.call(function()
        if type(flagName) == THValueType.STRING then
            local flagId = flagName:upper()
            if flagId == "SELF" then
                flagName = self.debugFlagId
            end
        end
    end)
    return THDebugUtil.consoleCommandSetDebugEnabled(flagName, isEnabled, sendToServer)
end