--[[
Copyright (C) GtX (Andy), 2021

Author: GtX | Andy
Date: 12.04.2021
Revision: FS25-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]

SupplyTransportMission = {}

SupplyTransportMission.MOD_NAME = g_currentModName
SupplyTransportMission.MOD_DIRECTORY = g_currentModDirectory

SupplyTransportMission.NAME = "supplyTransportMission"

SupplyTransportMission.MISSION_FAILED_FACTOR = 0.3
SupplyTransportMission.DEFAULT_MAX_NUM_INSTANCES = 5
SupplyTransportMission.MAX_NUM_INSTANCES = 15 -- Can be increased from default 5 to maximum 15 using console command or in the save game :-)

SupplyTransportMission.DEBUG_ENABLED = false
SupplyTransportMission.EMPTY_TABLE = {}

local SupplyTransportMission_mt = Class(SupplyTransportMission, AbstractMission)
InitObjectClass(SupplyTransportMission, "SupplyTransportMission")

function SupplyTransportMission.registerXMLPaths(schema, key)
    SupplyTransportMission:superClass().registerXMLPaths(schema, key)

    schema:register(XMLValueType.STRING, key .. ".invalidFillTypes#fillTypes", "Allow mod map creators the ability to exclude custom fill types from Supply and Transport missions.")
end

function SupplyTransportMission.registerSavegameXMLPaths(schema, key)
    SupplyTransportMission:superClass().registerSavegameXMLPaths(schema, key)

    schema:register(XMLValueType.INT, key .. ".contract#npcIndex", "NPC index used")
    schema:register(XMLValueType.STRING, key .. ".contract#fillType", "Name of the fill type")
    schema:register(XMLValueType.FLOAT, key .. ".contract#expectedLiters", "Expected litres")
    schema:register(XMLValueType.FLOAT, key .. ".contract#depositedLiters", "Deposited litres")
    schema:register(XMLValueType.FLOAT, key .. ".contract#pricePerLitre", "Price per litre")

    schema:register(XMLValueType.STRING, key .. ".sellingStation#placeableUniqueId", "Unique id of the selling point")
    schema:register(XMLValueType.INT, key .. ".sellingStation#unloadingStationIndex", "Index of the unloading station")
end

function SupplyTransportMission.registerMetaXMLPaths(schema, key)
    SupplyTransportMission:superClass().registerMetaXMLPaths(schema, key)

    schema:register(XMLValueType.INT, key .. "#maxNumInstances", "The user set maximum instances that can be available at one time. (SP Console Command Only for version 1.0.0.0)")
end

function SupplyTransportMission.new(isServer, isClient, customMt)
    local title = g_i18n:getText("contract_supplyTransport_title", SupplyTransportMission.MOD_NAME)
    local description = g_i18n:getText("contract_supplyTransport_defaultDescription", SupplyTransportMission.MOD_NAME)
    local self = AbstractMission.new(isServer, isClient, title, description, customMt or SupplyTransportMission_mt)

    self.customEnvironment = SupplyTransportMission.MOD_NAME
    self.extraProgressText = g_i18n:getText("contract_supplyTransport_extraProgress", self.customEnvironment)
    self.missionProgressText = g_i18n:getText("contract_supplyTransport_missionProgress", self.customEnvironment)

    self.sellingStationRemoved = false
    self.pendingSellingStationId = nil
    self.sellingStation = nil

    self.fillTypeIndex = nil
    self.pricePerLitre = 0

    self.depositedLiters = 0
    self.expectedLiters = 0

    self.reimbursementPerHa = 0
    self.lastFillSoldTime = -1

    self.sellingStationPlaceableUniqueId = uniqueId
    self.unloadingStationIndex = unloadingStationIndex

    self.npcIndex = 1
    self.farmlandId = FarmlandManager.NO_OWNER_FARM_ID

    self.mapHotspots = {}

    self.addSellingStationHotspot = false

    return self
end

function SupplyTransportMission:init(fillTypeIndex, missionType)
    if not SupplyTransportMission:superClass().init(self) or (fillTypeIndex == nil) then
        return false
    end

    local sellingStation, pricePerLitre = self:getRandomSellPoint(fillTypeIndex)

    if sellingStation == nil or pricePerLitre <= 0 then
        return false
    end

    self.fillTypeIndex = fillTypeIndex
    self.pricePerLitre = pricePerLitre

    self.depositedLiters = 0
    self.expectedLiters = SupplyTransportMission.getRandomLitres(fillTypeIndex, missionType)
    self.npcIndex = SupplyTransportMission.getRandomNpcIndex(missionType)

    self.contractDuration = (math.random(10, 480) / 10) + math.random()
    self.reward = self.expectedLiters * (pricePerLitre * SupplyTransportMission.getRewardMultiplier())

    self:setSellingStation(sellingStation)

    return true
end

function SupplyTransportMission:delete()
    if self.sellingStation ~= nil then
        self.sellingStation.missions[self] = nil
    end

    if self.sellingStationMapHotspot ~= nil then
        table.removeElement(self.mapHotspots, self.sellingStationMapHotspot)
        g_currentMission:removeMapHotspot(self.sellingStationMapHotspot)

        self.sellingStationMapHotspot:delete()
        self.sellingStationMapHotspot = nil

        self.addSellingStationHotspot = false
    end

    SupplyTransportMission:superClass().delete(self)
end

function SupplyTransportMission:update(dt)
    SupplyTransportMission:superClass().update(self, dt)

    if self.status == MissionStatus.RUNNING and not self.addSellingStationHotspot then
        if g_localPlayer ~= nil and g_localPlayer.farmId == self.farmId then
            self:addHotspot()
        end
    end

    if self.pendingSellingStationId ~= nil then
        self:tryToResolveSellingStation()
    end

    if self.isServer then
        if self.lastFillSoldTime > 0 then
            self.lastFillSoldTime = self.lastFillSoldTime - 1

            if self.lastFillSoldTime == 0 then
                local percent = math.floor((self.completion * 100) + 0.5)
                g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, string.format(self.missionProgressText, percent, self:getFillTypeTitle()))
            end
        end
    end
end

function SupplyTransportMission:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.sellingStation)

    streamWriteUIntN(streamId, self.fillTypeIndex, FillTypeManager.SEND_NUM_BITS)
    streamWriteUInt16(streamId, math.floor(self.pricePerLitre * 1000 + 0.5))

    streamWriteFloat32(streamId, self.expectedLiters)

    if streamWriteBool(streamId, self.depositedLiters > 0) then
        streamWriteFloat32(streamId, self.depositedLiters)
    end

    streamWriteUInt8(streamId, self.npcIndex or 1)
    streamWriteBool(streamId, self.sellingStationRemoved == true)

    SupplyTransportMission:superClass().writeStream(self, streamId, connection)
end

function SupplyTransportMission:readStream(streamId, connection)
    self.pendingSellingStationId = NetworkUtil.readNodeObjectId(streamId)

    self.fillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)
    self.pricePerLitre = streamReadUInt16(streamId) / 1000

    self.expectedLiters = streamReadFloat32(streamId)

    if streamReadBool(streamId) then
        self.depositedLiters = streamReadFloat32(streamId)
    end

    self.npcIndex = streamReadUInt8(streamId)
    self.sellingStationRemoved = streamReadBool(streamId)

    SupplyTransportMission:superClass().readStream(self, streamId, connection)
end

function SupplyTransportMission:writeUpdateStream(streamId, connection, dirtyMask)
    SupplyTransportMission:superClass().writeUpdateStream(self, streamId, connection, dirtyMask)

    if streamWriteBool(streamId, self.depositedLiters > 0) then
        streamWriteFloat32(streamId, self.depositedLiters)
    end

    streamWriteBool(streamId, self.sellingStationRemoved == true)
end

function SupplyTransportMission:readUpdateStream(streamId, timestamp, connection)
    SupplyTransportMission:superClass().readUpdateStream(self, streamId, timestamp, connection)

    if streamReadBool(streamId) then
        self.depositedLiters = streamReadFloat32(streamId)
    end

    self.sellingStationRemoved = streamReadBool(streamId)
end

function SupplyTransportMission.loadMetaDataFromXMLFile(xmlFile, key)
    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data ~= nil then
        data.maxNumInstances = math.clamp(xmlFile:getValue(key .. "#maxNumInstances", SupplyTransportMission.DEFAULT_MAX_NUM_INSTANCES), 1, SupplyTransportMission.MAX_NUM_INSTANCES)
    end
end

function SupplyTransportMission.saveMetaDataToXMLFile(xmlFile, key)
    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data ~= nil then
        if data.maxNumInstances ~= nil and data.maxNumInstances ~= SupplyTransportMission.DEFAULT_MAX_NUM_INSTANCES then
            xmlFile:setValue(key .. "#maxNumInstances", data.maxNumInstances)
        end
    end
end

function SupplyTransportMission:saveToXMLFile(xmlFile, key)
    xmlFile:setValue(key .. ".contract#npcIndex", self.npcIndex)

    xmlFile:setValue(key .. ".contract#fillType", g_fillTypeManager:getFillTypeNameByIndex(self.fillTypeIndex))
    xmlFile:setValue(key .. ".contract#expectedLiters", self.expectedLiters)
    xmlFile:setValue(key .. ".contract#depositedLiters", self.depositedLiters)
    xmlFile:setValue(key .. ".contract#pricePerLitre", self.pricePerLitre)

    local sellingStation = self.sellingStation

    if sellingStation ~= nil then
        local owningPlaceable = sellingStation.owningPlaceable

        if owningPlaceable ~= nil then
            local index = g_currentMission.storageSystem:getPlaceableUnloadingStationIndex(owningPlaceable, sellingStation)

            if index ~= nil then
                xmlFile:setValue(key .. ".sellingStation#placeableUniqueId", owningPlaceable:getUniqueId())
                xmlFile:setValue(key .. ".sellingStation#unloadingStationIndex", index)

                SupplyTransportMission:superClass().saveToXMLFile(self, xmlFile, key)
            else
                Logging.xmlWarning(xmlFile, "Failed to retrieve index of sellingStation '%s' at '%s.sellingStation'", self:getSellingStationName(), key)
            end
        else
            Logging.xmlWarning(xmlFile, "Failed to locate owningPlaceable for sellingStation '%s' at '%s.sellingStation'", self:getSellingStationName(), key)
        end
    else
        Logging.xmlWarning(xmlFile, "Mission does not contain a valid sellingStation at '%s.sellingStation'", key)
    end
end

function SupplyTransportMission:loadFromXMLFile(xmlFile, key)
    -- This is used to fix incorrectly checked nil of 'self.vehiclesToLoad' when loading from save game, this should only have been done when 'self.spawnedVehicles' was true @Giants.
    self.isLoadingFromSavegameXML = true

    if not SupplyTransportMission:superClass().loadFromXMLFile(self, xmlFile, key) then
        return false
    end

    if self.vehiclesToLoad == nil then
        self.tryToAddMissingVehicles = false -- Just in case another update changes another 6 year old BOOL value for no reason.
    end

    local fillTypeName = xmlFile:getValue(key .. ".contract#fillType")
    local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName)

    if fillType == nil then
        Logging.xmlError(xmlFile, "FillType with name '%s' is not defined at '%s'", fillTypeName, key)

        return false
    end

    local expectedLiters = xmlFile:getValue(key .. ".contract#expectedLiters", 0)

    if expectedLiters <= 0 then
        Logging.xmlError(xmlFile, "Failed to retrieve valid 'expectedLiters' at '%s'", key)

        return false
    end

    local npcIndex = xmlFile:getValue(key .. ".contract#npcIndex")

    if npcIndex == nil or g_npcManager:getNPCByIndex(npcIndex) == nil then
        npcIndex = SupplyTransportMission.getRandomNpcIndex()
        Logging.xmlWarning(xmlFile, "Failed to retrieve valid NPC index from saved mission, trying to assign a new one. (%s)", key)
    end

    self.npcIndex = npcIndex
    self.fillTypeIndex = fillType.index
    self.expectedLiters = expectedLiters

    self.depositedLiters = xmlFile:getValue(key .. ".contract#depositedLiters", self.depositedLiters)
    self.pricePerLitre = xmlFile:getValue(key .. ".contract#pricePerLitre", self.pricePerLitre)

    local uniqueId = xmlFile:getValue(key .. ".sellingStation#placeableUniqueId")

    if uniqueId == nil then
        Logging.xmlError(xmlFile, "No sellingStation 'placeableUniqueId' given for Supply & Transport mission at '%s.sellingStation'", key)

        return false
    end

    local unloadingStationIndex = xmlFile:getValue(key .. ".sellingStation#unloadingStationIndex")

    if unloadingStationIndex == nil then
        Logging.xmlError(xmlFile, "No 'unloadingStationIndex' given for Supply & Transport mission at '%s.sellingStation'", key)

        return false
    end

    self.sellingStationPlaceableUniqueId = uniqueId
    self.unloadingStationIndex = unloadingStationIndex

    self.vehiclesToLoad = nil
    self.isLoadingFromSavegameXML = false

    return true
end

function SupplyTransportMission:onSavegameLoaded()
    if self.sellingStationPlaceableUniqueId == nil or self.unloadingStationIndex == nil then
        return
    end

    local placeable = g_currentMission.placeableSystem:getPlaceableByUniqueId(self.sellingStationPlaceableUniqueId)

    if placeable ~= nil then
        local sellingStation = g_currentMission.storageSystem:getPlaceableUnloadingStation(placeable, self.unloadingStationIndex)

        if sellingStation ~= nil then
            if self.pricePerLitre <= 0 then
                -- Try and resolve the price, if not then just destroy the mission
                self.pricePerLitre = sellingStation:getEffectiveFillTypePrice(self.fillTypeIndex)

                if self.pricePerLitre == nil then
                    Logging.error("[SupplyTransportMission] Failed to retrieve a valid 'pricePerLitre' from save game")
                    g_missionManager:markMissionForDeletion(self)

                    return
                end

                Logging.devInfo("[SupplyTransportMission] Resolved missing 'pricePerLitre' for mission with uniqueId %s", self.uniqueId)
            end

            self:setSellingStation(sellingStation)

            if self:getWasStarted() and not self:getIsFinished() then
                sellingStation.missions[self] = self
            end

            SupplyTransportMission:superClass().onSavegameLoaded(self)
        else
            Logging.error("[SupplyTransportMission] Failed to retrieve unloadingStation with index '%d' for placeable sellingStation '%s'", self.unloadingStationIndex, placeable.configFileName)
            g_missionManager:markMissionForDeletion(self)
        end
    else
        Logging.error("[SupplyTransportMission] Selling station placeable with uniqueId '%s' no longer available", self.sellingStationPlaceableUniqueId)
        g_missionManager:markMissionForDeletion(self)
    end
end

function SupplyTransportMission:updateTexts()
    local fillTypeTitle = self:getFillTypeTitle()

    if not string.isNilOrWhitespace(fillTypeTitle) and self.expectedLiters > 0 then
        -- Fill Types both in mods and base game are inconsistent with the naming. Sometimes it is pieces as it should be and other times litres when it should be pieces. Not sure there is much I can do?
        local litres = string.format("%s %s", g_i18n:formatNumber(g_i18n:getVolume(self.expectedLiters), 0), g_i18n:getVolumeUnit(true))

        self.title = g_i18n:getText("contract_supplyTransport_title", self.customEnvironment)
        self.progressTitle = string.format("%s ( %s )", self.title, fillTypeTitle)
        self.description = string.format(g_i18n:getText("contract_supplyTransport_description", self.customEnvironment), litres, fillTypeTitle, self:getSellingStationName())
    end
end

function SupplyTransportMission:getTitle()
    self:updateTexts()

    return self.title
end

function SupplyTransportMission:getDescription()
    self:updateTexts()

    if self.sellingStationRemoved and self.finishState == MissionFinishState.FAILED then
        return `{self.description}\n\n{g_i18n:getText("ai_validationErrorUnloadingStationDoesNotExistAnymore")}`
    end

    if not self:getWasStarted() then
        return `{self.description} {self.extraProgressText}`
    end

    return self.description
end

function SupplyTransportMission:getExtraProgressText()
    return self.extraProgressText or ""
end

function SupplyTransportMission:getVehicleGroupFromIdentifier(identifier, ...)
    local vehicles, rewardScale, errorMessage, group = SupplyTransportMission:superClass().getVehicleGroupFromIdentifier(self, identifier, ...)

    if self.isLoadingFromSavegameXML and vehicles == nil then
        vehicles = {} -- Fix for patch 1.14.0.0 that again breaks this mod, which is depressing but out of my control.
    end

    return vehicles, rewardScale, errorMessage, group
end

function SupplyTransportMission:hasLeasableVehicles()
    return false -- No need for vehicles, the point of the missions is to have the crop and equipment
end

function SupplyTransportMission:tryToResolveSellingStation()
    if self.pendingSellingStationId ~= nil and self.sellingStation == nil then
        self:setSellingStation(NetworkUtil.getObject(self.pendingSellingStationId))
    end
end

function SupplyTransportMission:setSellingStation(sellingStation)
    if sellingStation == nil then
        -- Logging.devError("[SupplyTransportMission] Failed to set sellingStation, value was nil")

        return
    end

    self.pendingSellingStationId = nil
    self.sellingStation = sellingStation

    self:updateTexts()

    local placeable = sellingStation.owningPlaceable

    if placeable ~= nil and placeable.getHotspot ~= nil then
        local hotspot = placeable:getHotspot()

        if hotspot ~= nil then
            self.sellingStationMapHotspot = HarvestMissionHotspot.new()
            self.sellingStationMapHotspot:setWorldPosition(hotspot:getWorldPosition())

            table.addElement(self.mapHotspots, self.sellingStationMapHotspot)

            -- Should only be true if mission is already active
            if self.addSellingStationHotspot then
                g_currentMission:addMapHotspot(self.sellingStationMapHotspot)
            end
        end
    end
end

function SupplyTransportMission:addHotspot()
    if self.sellingStationMapHotspot ~= nil then
        g_currentMission:addMapHotspot(self.sellingStationMapHotspot)
    end

    self.addSellingStationHotspot = true
end

function SupplyTransportMission:removeHotspot()
    if self.sellingStationMapHotspot ~= nil then
        g_currentMission:removeMapHotspot(self.sellingStationMapHotspot)
    end

    self.addSellingStationHotspot = false
end

function SupplyTransportMission:start()
    if self.pendingSellingStationId ~= nil then
        self:tryToResolveSellingStation()
    end

    if self.sellingStation == nil or not SupplyTransportMission:superClass().start(self) then
        return false
    end

    self.sellingStation.missions[self] = self

    return true
end

function SupplyTransportMission:finish(finishState)
    if self.sellingStation ~= nil then
        self.sellingStation.missions[self] = nil
    end

    self:removeHotspot()

    if g_currentMission:getFarmId() == self.farmId then
        local fillTypeTitle = self:getFillTypeTitle()

        if finishState == MissionFinishState.SUCCESS then
            g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_OK, string.format(g_i18n:getText("contract_supplyTransport_finished", self.customEnvironment), fillTypeTitle))
        else
            self.reimbursement = (self.pricePerLitre or 0) * self.depositedLiters -- Will be recalculated but do it now so client can have correct value in GUI

            if finishState == MissionFinishState.FAILED then
                if not self.sellingStationRemoved then
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, string.format(g_i18n:getText("contract_supplyTransport_failed", self.customEnvironment), fillTypeTitle))
                else
                    g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, string.format(g_i18n:getText("contract_supplyTransport_sellingStationRemoved", self.customEnvironment), fillTypeTitle))
                end
            elseif finishState == MissionFinishState.TIMED_OUT then
                g_currentMission:addIngameNotification(FSBaseMission.INGAME_NOTIFICATION_CRITICAL, string.format(g_i18n:getText("contract_supplyTransport_timedOut", self.customEnvironment), fillTypeTitle))
            end
        end
    end

    SupplyTransportMission:superClass().finish(self, finishState)
end

function SupplyTransportMission:dismiss()
    self.status = MissionStatus.DISMISSED

    if self.isServer then
        self:calculateReimbursement()

        local reward = self:getTotalReward()

        if reward ~= 0 then
            g_currentMission:addMoney(reward, self.farmId, MoneyType.MISSIONS, true, true)
        end
    end
end

function SupplyTransportMission:validate()
    return SupplyTransportMission:superClass().validate(self) and not self.sellingStationRemoved
end

function SupplyTransportMission:fillSold(delta)
    self.depositedLiters = math.min(self.depositedLiters + delta, self.expectedLiters)

    if self.depositedLiters >= self.expectedLiters then
        if self.sellingStation ~= nil then
            self.sellingStation.missions[self] = nil -- Remove mission from the Selling Station and start selling
        end
    end

    self.lastFillSoldTime = 30 -- Reset notification timer
end

function SupplyTransportMission:onSellingStationRemoved(sellingStation)
    if self.sellingStation == sellingStation then
        self.sellingStationRemoved = true

        if self.isServer then
            self:raiseDirtyFlags(self.missionDirtyFlag)
        end
    end
end

function SupplyTransportMission:getDetails()
    local details = SupplyTransportMission:superClass().getDetails(self)

    if self.pendingSellingStationId ~= nil then
        self:tryToResolveSellingStation()
    end

    local i18n = g_i18n
    local isEnglish = g_languageShort == "en" -- Used to correct some English translation issues
    local unitText = i18n:getVolumeUnit(false) -- Use short version to match other mission types

    table.insert(details, {
        title = isEnglish and "Selling station" or i18n:getText("contract_details_harvesting_sellingStation"),
        value = self:getSellingStationName()
    })

    table.insert(details, {
        title = isEnglish and "Fill type" or i18n:getText("infohud_fillType"),
        value = self:getFillTypeTitle()
    })

    table.insert(details, {
        title = i18n:getText("contract_total"),
        value = i18n:formatVolume(self.expectedLiters, 0, unitText)
    })

    if self:getWasStarted() then
        table.insert(details, {
            title = i18n:getText("contract_progress"),
            value = i18n:formatVolume(self.depositedLiters, 0, unitText)
        })
    end

    local minutesLeft = self:getMinutesLeft()

    table.insert(details, {
        title = i18n:getText("ui_pendingMissionTimeLeftTitle"),
        value = i18n:formatMinutes(minutesLeft > 0 and minutesLeft or nil)
    })

    return details
end

function SupplyTransportMission:getCompletion()
    if self.expectedLiters <= 0 then
        return 0
    end

    return math.max(math.min(1, self.depositedLiters / self.expectedLiters), 0.00001) -- Fix for another base game change in update v1.5 that could have been avoided.
end

function SupplyTransportMission:getLocation()
    return self:getFillTypeTitle()
end

function SupplyTransportMission:getReward()
    return self.reward or 0
end

function SupplyTransportMission.getRewardMultiplier()
    if g_currentMission.missionInfo.economicDifficulty ~= EconomicDifficulty.HARD then
        return 1.4
    end

    return 1.2
end

function SupplyTransportMission:getNPC()
    return g_npcManager:getNPCByIndex(self.npcIndex)
end

function SupplyTransportMission:getFarmlandId()
    return self.farmlandId or FarmlandManager.NO_OWNER_FARM_ID -- GUI does not do a nil check so just use 0
end

function SupplyTransportMission:getMapHotspots()
    return self.mapHotspots
end

function SupplyTransportMission:getFillTypeTitle()
    if self.fillTypeTitle == nil then
        local fillTypeDesc = g_fillTypeManager:getFillTypeByIndex(self.fillTypeIndex)

        if fillTypeDesc ~= nil then
            self.fillTypeTitle = fillTypeDesc.title
        end
    end

    return self.fillTypeTitle or "Unknown"
end

function SupplyTransportMission:getSellingStationName()
    local sellingStation = self.sellingStation

    if sellingStation ~= nil then
        if sellingStation.getName ~= nil then
            return sellingStation:getName() or "Unknown"
        elseif sellingStation.owningPlaceable ~= nil and sellingStation.owningPlaceable.getName ~= nil then
            return owningPlaceable:getName() or "Unknown"
        end
    end

    return "Unknown"
end

function SupplyTransportMission:getMissionTypeName()
    return SupplyTransportMission.NAME
end

function SupplyTransportMission:getStealingCosts()
    if self.stealingCost == 0 then
        self:calculateStealingCost()
    end

    return self.stealingCost or 0
end

function SupplyTransportMission:calculateStealingCost()
    if self.isServer then
        local stealingCost = 0

        if self.finishState ~= MissionFinishState.SUCCESS then
            if not self.sellingStationRemoved and self.pricePerLitre ~= nil then
                stealingCost = self.pricePerLitre * (self.expectedLiters - self.depositedLiters) * SupplyTransportMission.MISSION_FAILED_FACTOR
            end
        end

        self.stealingCost = stealingCost
    end

    return self.stealingCost
end

function SupplyTransportMission:calculateReimbursement()
    local reimbursement = 0

    if self.finishState ~= MissionFinishState.SUCCESS then
        reimbursement = (self.pricePerLitre or 0) * self.depositedLiters
    end

    self.reimbursement = reimbursement

    return reimbursement
end

function SupplyTransportMission:getRandomSellPoint(fillTypeIndex)
    local numSellPoints = 0
    local sellPoints = {}

    for _, unloadingStation in pairs(g_currentMission.storageSystem:getUnloadingStations()) do
        if unloadingStation.isSellingPoint and unloadingStation.allowMissions and unloadingStation.owningPlaceable ~= nil and unloadingStation.acceptedFillTypes[fillTypeIndex] then
            local pricePerLitre = unloadingStation:getEffectiveFillTypePrice(fillTypeIndex)

            if pricePerLitre > 0 then
                table.insert(sellPoints, {
                    sellingStation = unloadingStation,
                    pricePerLitre = pricePerLitre
                })

                numSellPoints += 1
            end
        end
    end

    if numSellPoints > 0 then
        local index = numSellPoints > 1 and math.random(numSellPoints) or 1
        local sellPoint = sellPoints[index]

        return sellPoint.sellingStation, sellPoint.pricePerLitre
    end

    return nil, 0
end

function SupplyTransportMission.getRandomFilltype(missionType)
    if missionType == nil then
        missionType = g_missionManager:getMissionType(SupplyTransportMission.NAME)
    end

    local data = missionType.data
    local validFillTypes, ignoreFillTypes = {}, {}
    local cropFillTypeMissions, addCropFillTypesOnly = 0, true

    local invalidFillTypes = data.invalidFillTypes or SupplyTransportMission.EMPTY_TABLE
    local userExcludedFillTypes = data.userExcludedFillTypes or SupplyTransportMission.EMPTY_TABLE
    local missions = g_missionManager:getMissionsByType(missionType.typeId)

    if missions ~= nil then
        for _, mission in ipairs (missions) do
            if g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(mission.fillTypeIndex) ~= nil then
                cropFillTypeMissions += 1
            end

            ignoreFillTypes[mission.fillTypeIndex] = true
        end
    end

    local function getIsValidMissionFillType(fillTypeIndex)
        if ignoreFillTypes[fillTypeIndex] or invalidFillTypes[fillTypeIndex] then
            return false
        end

        return userExcludedFillTypes[fillTypeIndex] == nil
    end

    addCropFillTypesOnly = cropFillTypeMissions < data.maxNumInstances * 0.5 -- Make sure half of the missions are crop fillTypes

    for _, unloadingStation in pairs(g_currentMission.storageSystem:getUnloadingStations()) do
        if unloadingStation.isSellingPoint and unloadingStation.allowMissions and unloadingStation.owningPlaceable ~= nil then
            for fillTypeIndex, _ in pairs (unloadingStation.acceptedFillTypes) do
                if getIsValidMissionFillType(fillTypeIndex) then
                    ignoreFillTypes[fillTypeIndex] = true

                    if not addCropFillTypesOnly or g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(fillTypeIndex) ~= nil then
                        table.insert(validFillTypes, fillTypeIndex)
                    end
                end
            end
        end
    end

    local numValidFillTypes = #validFillTypes

    if numValidFillTypes > 0 then
        if numValidFillTypes == 1 then
            return validFillTypes[1]
        end

        if numValidFillTypes == 2 then
            return validFillTypes[math.random() < 0.5 and 1 or 2]
        end

        Utils.shuffle(validFillTypes)

        return validFillTypes[math.random(numValidFillTypes)]
    end

    return nil
end

function SupplyTransportMission.getRandomLitres(fillTypeIndex, missionType)
    if missionType == nil then
        missionType = g_missionManager:getMissionType(SupplyTransportMission.NAME)
    end

    if fillTypeIndex == FillType.COTTON then
        return math.random(1, 8) * 20000 -- 20000 > 160000
    end

    local maxValue = 250 -- Maximum of 250000 litres for any mission

    if g_fruitTypeManager:getFruitTypeIndexByFillTypeIndex(fillTypeIndex) == nil then
        maxValue = 125 -- Maximum of 125000 litres for non crop type missions
    end

    -- If not cotton then make sure there are missions 50000 litres or less
    local missions = g_missionManager:getMissionsByType(missionType.typeId)

    if missions ~= nil then
        local numSmallMission = 0
        local halfMaxNumInstances = missionType.data.maxNumInstances * 0.5

        for _, mission in ipairs (missions) do
            if mission.expectedLiters <= 50000 then
                numSmallMission += 1

                if numSmallMission >= halfMaxNumInstances then
                    return math.random(5, maxValue) * 1000 -- 5000 > 250000
                end
            end
        end
    end

    return math.random(5, 50) * 1000 -- 5000 > 50000
end

function SupplyTransportMission.getRandomNpcIndex(missionType)
    if missionType == nil then
        missionType = g_missionManager:getMissionType(SupplyTransportMission.NAME)
    end

    local validNPCS, numValidNpcs = {}, 0
    local missions = g_missionManager:getMissionsByType(missionType.typeId)

    if missions ~= nil then
        local function getCanAddNPC(npcIndex)
            for _, mission in ipairs (missions) do
                if mission.npcIndex == npcIndex then
                    return false
                end
            end

            return true
        end

        for index in ipairs(g_npcManager.npcs) do
            if getCanAddNPC(index) then
                table.insert(validNPCS, index)
            end
        end

        numValidNpcs = #validNPCS
    end

    if numValidNpcs > 0 then
        if numValidNpcs == 1 then
            return validNPCS[1]
        end

        return validNPCS[math.random(numValidNpcs)]
    end

    return g_npcManager:getRandomIndex()
end

function SupplyTransportMission.tryGenerateMission()
    local missionType = g_missionManager:getMissionType(SupplyTransportMission.NAME)

    if SupplyTransportMission.canRun(missionType.data) then
        local fillTypeIndex = SupplyTransportMission.getRandomFilltype(missionType)

        if fillTypeIndex ~= nil then
            local mission = SupplyTransportMission.new(true, g_client ~= nil)

            if mission:init(fillTypeIndex, missionType) then
                local environment = g_currentMission.environment
                local durationMs = MathUtil.hoursToMs(mission.contractDuration)
                local endDay, endDayTime = environment:getDayAndDayTime(environment.dayTime + durationMs, environment.currentMonotonicDay)

                mission:setEndDate(endDay, endDayTime)

                -- TODO: Add to debugger
                if SupplyTransportMission.DEBUG_ENABLED then
                    SupplyTransportMission.debugGeneration(mission)
                end

                return mission
            else
                mission:delete()
            end
        end
    end

    return nil
end

function SupplyTransportMission.loadMapData(xmlFile, key, baseDirectory)
    if g_supplyTransportManager ~= nil then
        g_supplyTransportManager:loadMapData(xmlFile, key, baseDirectory)
    end

    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data ~= nil then
        if data.defaultFillTypes == nil then
            data.defaultFillTypes = {} -- Do not overwrite manager
        end

        if data.invalidFillTypes == nil then
            data.invalidFillTypes = {} -- Do not overwrite manager
        end

        data.mapInvalidFillTypes = {} -- Should not exist at this point so create new table

        local fillTypeManager = g_fillTypeManager
        local numMapInvalidFillTypes = 0
        local numInvalidFillTypes = 0

        local function addInvalidFillType(fillTypeIndex, isMapSet)
            if data.defaultFillTypes[fillTypeIndex] ~= nil or data.invalidFillTypes[fillTypeIndex] ~= nil then
                return
            end

            data.invalidFillTypes[fillTypeIndex] = true
            numInvalidFillTypes += 1

            if isMapSet then
                data.mapInvalidFillTypes[fillTypeIndex] = true
                numMapInvalidFillTypes += 1
            end
        end

        -- Load invalid fill types to avoid errors on base game and mod maps
        local configXmlPath = SupplyTransportMission.MOD_DIRECTORY .. "shared/supplyTransportMission.xml"
        local configXmlFile = XMLFile.loadIfExists("supplyTransportConfigXML", configXmlPath, Mission00.xmlSchema)

        if configXmlFile ~= nil then
            local invalidFillTypeNames = configXmlFile:getValue("map.missions.supplyTransportMission.invalidFillTypes#fillTypes")

            if invalidFillTypeNames ~= nil then
                for _, fillTypeIndex in pairs(fillTypeManager:getFillTypesByNames(invalidFillTypeNames, nil)) do
                    addInvalidFillType(fillTypeIndex, false)
                end
            end

            configXmlFile:delete()
        end

        -- Check if a mod map has invalidated custom fill types
        if xmlFile:hasProperty(key) then
            local invalidFillTypeNames = xmlFile:getValue(key .. ".invalidFillTypes#fillTypes")

            if invalidFillTypeNames ~= nil then
                for _, fillTypeIndex in pairs(fillTypeManager:getFillTypesByNames(invalidFillTypeNames, nil)) do
                    addInvalidFillType(fillTypeIndex, true)
                end
            end
        end

        if numMapInvalidFillTypes > 0 then
            local mapInvalidFillTypes = {}

            for fillTypeIndex, _ in pairs(data.mapInvalidFillTypes) do
                local fillType = fillTypeManager:getFillTypeByIndex(fillTypeIndex)

                if fillType ~= nil then
                    table.insert(mapInvalidFillTypes, fillType.name)
                end
            end

            if #mapInvalidFillTypes > 0 then
                Logging.info("[SupplyTransportMission] Mod map has added the following invalid fill types: %s", table.concat(mapInvalidFillTypes, ", "))
            end
        end

        -- Invalidate mod bales
        for _, fillType in ipairs(fillTypeManager:getFillTypes()) do
            if data.invalidFillTypes[fillType.index] == nil then
                if fillType.name:startsWith("ROUNDBALE_") or fillType.name:startsWith("SQUAREBALE_") then
                    addInvalidFillType(fillType.index, false)
                end
            end
        end

        -- TODO: Add to debugger
        if numInvalidFillTypes > 0 and SupplyTransportMission.DEBUG_ENABLED then
            for fillTypeIndex, _ in pairs(data.invalidFillTypes) do
                local fillType = fillTypeManager:getFillTypeByIndex(fillTypeIndex)

                if fillType ~= nil then
                    Logging.info("[SupplyTransportMission] Ignoring invalid fillType (%d) | (%s) | (%s)", fillTypeIndex, fillType.name, fillType.title)
                end
            end
        end
    else
        Logging.devInfo("[SupplyTransportMission] Failed to load map data!")
    end

    return true
end

function SupplyTransportMission.canRun(data)
    if data == nil then
        data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)
    end

    if data == nil or data.numInstances == nil or data.maxNumInstances == nil then
        return false
    end

    return data.numInstances < data.maxNumInstances
end

function SupplyTransportMission.debugGeneration(mission)
    if mission == nil or g_currentMission == nil then
        return
    end

    local environment = g_currentMission.environment
    local currentTime = environment.dayTime / 3600000
    local timeHours = math.floor(currentTime)
    local finishTime = mission.endDate.endDayTime / 3600000
    local finishTimeHours = math.floor(finishTime)

    local sellingStation = string.format("    - Selling Station: %s", mission:getSellingStationName())
    local fillType = string.format("    - Fill Type: %s ( %d )", mission:getFillTypeTitle(), mission.fillTypeIndex or 0)
    local expected = string.format("    - Expected Litres: %s", mission.expectedLiters)
    local price = string.format("    - Price Per Litre: %s ( %s )", g_i18n:formatMoney((mission.pricePerLitre or 0) * 1000, 0, true, true), mission.pricePerLitre or 0)
    local reward = string.format("    - Reward: %s", g_i18n:formatMoney(mission.reward, 0, true, true))
    local profit = string.format("    - Profit: %s", g_i18n:formatMoney(mission.reward - (mission.expectedLiters * mission.pricePerLitre), 0, true, true))
    local duration = string.format("    - Duration: %s ( %s )", g_i18n:formatMinutes(mission:getMinutesLeft()), mission.contractDuration)
    local currentDay = string.format("    - Current Day: %s", environment.currentMonotonicDay)
    local currentDayTime = string.format("    - Current Day Time: %02d:%02d", timeHours, math.floor((currentTime - timeHours) * 60))
    local finishDay = string.format("    - Finish Day: %s", mission.endDate.endDay)
    local finishDayTime = string.format("    - Finish Day Time: %02d:%02d ( %s )", finishTimeHours, math.floor((finishTime - finishTimeHours) * 60), mission.endDate.endDayTime)

    Logging.info("[SupplyTransportMission] Mission Generated.\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n", sellingStation, fillType, expected, price, reward, profit, duration, currentDay, currentDayTime, finishDay, finishDayTime)
end
